using System;
using System.IO;
using DynaPDF;

namespace examples
{
	class Class1
	{
      /*
         Note that the dynapdf.dll must be copied into the output directory or into a
         Windwos search path (e.g. %WINDOWS%/System32) before the application can be executed!
      */

      // Error callback function.
      // If the function name should not appear at the beginning of the error message then set
      // the flag emNoFuncNames (pdf.SetErrorMode(DynaPDF.TErrMode.emNoFuncNames);).
      static int PDFError(IntPtr Data, int ErrCode, IntPtr ErrMessage, int ErrType)
      {
         Console.Write("{0}\n", System.Runtime.InteropServices.Marshal.PtrToStringAnsi(ErrMessage));
         return 0; // We try to continue if an error occurrs. Any other return value breaks processing.
      }

      // Helper function
      static byte[] GetFileBuf(String FileName)
      {
         byte[] retval;
         System.IO.FileStream f = new System.IO.FileStream(FileName, FileMode.Open);
         
         retval = new byte[f.Length];
         f.Read(retval, 0, (int)f.Length);
         f.Close();
         return retval;
      }

      static bool HaveEInvoice(CPDF PDF, String InFileName)
      {
         bool retval = false;
         TPDFVersionInfo info = new TPDFVersionInfo();

         PDF.CreateNewPDF(null);
         // We need the document info or metadata and embedded files only
         PDF.SetImportFlags(TImportFlags.ifDocInfo | TImportFlags.ifEmbeddedFiles);
         PDF.SetImportFlags2(TImportFlags2.if2UseProxy);
         if (PDF.OpenImportFile(InFileName, TPwdType.ptOpen, null) < 0) goto finish;

         // Other stuff can be ignored
         PDF.ImportCatalogObjects();

         if (!PDF.GetPDFVersionEx(ref info)) goto finish;

         if (info.PDFAVersion == 3 && info.FXDocName != null)
         {
            int ef;
            TPDFFileSpec fs = new TPDFFileSpec();
            if ((ef = PDF.FindEmbeddedFile(info.FXDocName)) < 0)
            {
               Console.ForegroundColor = ConsoleColor.Red;
               Console.Write("Invoice {0} not found!\n", info.FXDocName);
               goto finish;
            }
            if (ef != 0)
            {
               Console.ForegroundColor = ConsoleColor.Yellow;
               Console.Write("Warning: The invoice should be the first file attachment. This might cause unnecessary problems.\n");
            }
            retval = (PDF.GetEmbeddedFile(ef, ref fs, true) && fs.BufSize > 0);
         }
        finish:
         PDF.FreePDF();
         return retval;
      }

      static bool CreateInvoice(CPDF PDF, bool FacturX, String InvoiceName, String OutFile)
      {
         int ef;
         bool retval = false;
         // The output file is opened later
         PDF.CreateNewPDF(null);

         // Set the license key here if you have one
         // PDF.SetLicenseKey("");

         // We assume that the PDF invoice is already a valid PDF/A 3 file in this example.

         PDF.SetImportFlags(TImportFlags.ifImportAsPage | TImportFlags.ifImportAll);
         if (PDF.OpenImportFile("../../../../test_files/test_invoice.pdf", TPwdType.ptOpen, null) < 0) goto finish;

         PDF.ImportPDFFile(1, 1.0, 1.0);

         /*
            The test invoice has the file name factur-x.xml but we must be able to override the name
            since the German XRechnung requires the name xrechnung.xml. With AttachFileEx() we can
            specify the file name. So, this is the right function for this test.
         */

         byte[] buffer = GetFileBuf("../../../../test_files/factur-x.xml");
         
         ef = PDF.AttachFileEx(buffer, buffer.Length, InvoiceName, "EN 19631 compliant invoice", false);
         PDF.AssociateEmbFile(TAFDestObject.adCatalog, -1, FacturX ? TAFRelationship.arAlternative : TAFRelationship.arSource, ef);

         /*
            Note that ZUGFeRD 2.1 or higher and FacturX is identically defined in PDF. Therefore, both formats share
            the same version constants. Note also that the profiles Minimum, Basic, and Basic WL are not fully EN 16931
            compliant, and hence cannot be used to create e-invoices.
         */
         PDF.SetPDFVersion(FacturX ? TPDFVersion.pvFacturX_Comfort : TPDFVersion.pvFacturX_XRechnung);

         // No fatal error occurred?
         if (PDF.HaveOpenDoc())
         {
            // OK, now we can open the output file.
            if (!PDF.OpenOutputFile(OutFile)) goto finish;
            retval = PDF.CloseFile();
         }

        finish:
         PDF.FreePDF();
         return retval;
      }

		[STAThread]
		static void Main(string[] args)
		{
         try
         {
            CPDF pdf = new CPDF();
            // Error messages and warnings are passed to the callback function.
            pdf.SetOnErrorProc(IntPtr.Zero, new DynaPDF.TErrorProc(PDFError));

            string filePath = Directory.GetCurrentDirectory() + "\\out.pdf";

            // Test cases:
            // - ZUGFeRD or FacturX
            // - XRechnung -> The invoice name must be xrechnung.xml
            if (!CreateInvoice(pdf, true,  "factur-x.xml",  filePath) || !HaveEInvoice(pdf, filePath)
            ||  !CreateInvoice(pdf, false, "xrechnung.xml", filePath) || !HaveEInvoice(pdf, filePath))
            {
               Console.ForegroundColor = ConsoleColor.Red;
               Console.Write("XML Invoice not found!");
            }else
            {
               Console.ForegroundColor = ConsoleColor.Green;
               Console.Write("All tests passed!");
            }
            pdf = null;
         }catch(Exception e)
         {
            Console.Write(e.Message);
         }
         Console.Read();
		}
	}
}
